******************************************************
* COUNT4.SRC                                         *
*                                                    *
* This program copies BCD data to four 7-segment     *
* displays driven by a MC14499 serial display        *
* driver.  A 4-digit count is copied at a rate of    *
* 10 Hz.  The interface uses the 68681 DUART to      *
* drive the MC14499 as follows:                      *
*                                                    *
*      68681                14499                    *
*      =====      ==============================     *
*       OP0       ENABLE (active low)                *
*       OP1       CLOCK (negative edge-triggered)    *
*       OP2       DATA                               *
******************************************************
DUART    EQU      $00C001     ;base address of 68681
OPR_SET  EQU      14*2        ;set bit command reg.
OPR_CLR  EQU      15*2        ;clear bit cmd. reg.
ONE_SEC  EQU      204800      ;count for 1 sec. delay

         ORG      $8000
COUNT4   MOVEA.L  #DUART,A0      ;A0 points to 68681
         MOVE.B   #7,OPR_CLR(A0) ;MC14499 inputs high
         MOVEA.L  #DIGITS,A1     ;A1 points to digits
         MOVEA.L  #INC,A2        ;A2 points to inc.
         MOVE.W   #0,(A1)        ;init BCD digits
         MOVE.B   #0,2(A1)       ;init decimal points
LOOP     BSR.S    UPDATE         ;update LED display
         LEA      2(A1),A1       ;A1 -> past digits
         LEA      2(A2),A2       ;A2 -> past increment
         ANDI.B   #$EF,CCR       ;clear X-bit in CCR
         ABCD     -(A2),-(A1)    ;increment BCD digits
         ABCD     -(A2),-(A1)
         MOVE.L   #ONE_SEC/10,D7 ;delay for 10 Hz
         BSR.S    DELAY
         BRA      LOOP

******************************************************
* UPDATE - send 4 BCD digits to MC14499              *
*                                                    *
* Clear OP0 (MC14499 ENABLE), send four decimal      *
* points = 0 followed by four digits, then set OP0.  *
* The most significant digit (bits 12-15) is sent    *
* first.                                             *
*                                                    *
*        ENTER - A0 points to 68681                  *
*              - A1 points to digits                 *
*        EXIT  - display updated, all reg. intact    *
*        USES  - OUT4                                *
******************************************************
UPDATE   MOVEM.W  D0-D1,-(SP)    ;save registers
         MOVE.B   #1,OPR_SET(A0) ;assert ENABLE line
         CLR.B    D0             ;clear decimal points
         BSR.S    OUT4           ;send DPs first
         MOVE.W   (A1),D0        ;get BCD digits
         MOVE.B   #4,D1          ;use D1 as counter
UPDATE2  ROL.W    #4,D0          ;align bits (0-3 1st)
         BSR.S    OUT4           ;send them
         SUBQ.B   #1,D1          ;last digit?
         BNE      UPDATE2        ;no:  send again
         MOVE.B   #1,OPR_CLR(A0) ;yes: deassert ENABLE
         MOVEM.W  (SP)+,D0-D1    ;restore registers
         RTS

******************************************************
* OUT4 - OUTput 4 bits to the MC14499                *
*                                                    *
* Place data bit on OP2 (MC14499 DATA), then toggle  *
* OP1 (MC14499 CLOCK).  Repeat four times.  Bit 3 is *
* sent first.  Serial data is output on OP2.         *
*                                                    *
*        ENTER:   D0[0:3] contains bits to output    *
*        EXIT:    data bits sent, all reg. intact    *
*        USES:    no subroutines                     *
******************************************************
OUT4     MOVEM.W  D0-D4,-(SP)
         ROR.B    #2,D0          ;align as D0[1,0,7,6]
         MOVE.B   #4,D3          ;use D3 as counter
LOOP2    ROL.B    #1,D0          ;put bit in D0 bit 2
         MOVE.B   D0,D2          ;save data
         ANDI.B   #$04,D0        ;mask other bits
         MOVE.B   D0,OPR_CLR(A0) ;if bit set, set pin
         EORI.B   #$04,D0        ;complement data bit
         MOVE.B   D0,OPR_SET(A0) ;if bit clr, clr pin
         MOVE.B   #$02,D4        ;create CLK pulse
         MOVE.B   D4,OPR_SET(A0) ;clear CLK pin
         NOP                     ;stretch (MC14499
         NOP                     ; needs 2 us pulse)
         MOVE.B   D4,OPR_CLR(A0) ;set CLK pin
         MOVE.B   D2,D0          ;restore data
         SUBQ     #1,D3          ;last of 4 bits?
         BNE      LOOP2          ;no:  do again
         MOVEM.W  (SP)+,D0-D4    ;yes: done
         RTS

******************************************************
* DELAY - create a software DELAY using D7 as count  *
*                                                    *
*        ENTER:   D7[0:31] = count                   *
*        EXIT:    D7[0:31] = 0                       *
*                 all other registers intact         *
*        USES:    no subroutines                     *
*                                                    *
* NOTE: If the 68000 is operating from a 3.6864 MHz  *
* crystal, the delay equals COUNT x 18/3686400 sec.  *
******************************************************
DELAY    SUBQ.L   #1,D7    ;8 cycles
         BNE      DELAY    ;10 cycles (branch taken)
         RTS

INC      DC.W     1        ;inc. = 1 (code segment)

         ORG      $A000    ;data segment at $A000
DIGITS   DS.W     1        ;digits in data segment
         END      COUNT4
